---
title: "Lucit Form Fields Specification"
description: "Complete specification for Lucit's JSON-based form field configuration system used for dynamic user input forms throughout the platform"
keywords: "lucit form fields, json configuration, field specification, dynamic forms, field validation"
---

# Lucit Form Fields Specification

## Overview

Lucit Form Fields provide a flexible, JSON-based configuration system for creating dynamic user input forms throughout the Lucit platform. These fields are used to configure settings for various features including data sources, apps, capabilities, automations, and more.

Fields are defined as an array of field objects stored in JSON format, with each field specifying its type, validation rules, display properties, and behavior.

## Usage

Form fields can be configured in the following places

- **Data Source Feed Params** The configurable settings for a data source - `User Profle -> Apps -> {AppName} -> Capabilities -> Data Source Application -> {GearIcon} -> Feed Params`
- **Capability Settings** For settings that support Capability Params, you can provide a set of configurable settings for the app that will appear to the user when adding / configuring the app. `User Profle -> Apps -> {AppName} -> Capabilities -> {SomeSupportedCapability} -> {GearIcon} -> EDIT CAPABILITY PARAMS`  
- **Automations** When building a manual automation that should request information from the user.  `User Profle -> Apps -> {AppName} -> Capabilities -> Automation -> Automations ->  {Automation} -> {PencilIcon} -> Form Fields (JSON)`    (Note: This appears for Manual automations only)
- **Custom Designer Element** When adding configurable options / settings to a custom designer element. `User Profle -> Apps -> {AppName} -> Capabilities -> Designer Templates and Elements -> {GearIcon} -> ELEMENT PACKS tab -> TEXT tab -> {Element} -> SETTINGS TAB`. These settings will appear when adding this element to a template canvas, and then right-clicking on the element and choosing `Edit Settings`. Learn more about the Template Designer in the [Template Designer General Navigation Guide](../../guides/template-designer-general-navigation-guide.md).

## Structure

Form fields are defined within a JSON object with the following top-level structure:

```json
{
  "fields": [
    // Array of field objects
  ],
  "secrets": {},
  "global_settings": {}
}
```

## Field Object Schema

### Required Properties

Every field must include these properties:

| Property | Type | Description |
|----------|------|-------------|
| `key` | string | Unique identifier for the field. Used to store and retrieve the field's value. |
| `type` | string | The data type of the field. See [Field Types](#field-types) for available options. |

**Note:** Most field types also require a `name` property (see [Optional Properties](#optional-properties)). The exceptions are `header`, `divider`, and `space` types which are display-only elements.

### Optional Properties

| Property | Type | Description |
|----------|------|-------------|
| `name` | string | Display label for the field. Required for all field types except `header`, `divider`, and `space`. |
| `description` | string | Help text displayed below or near the field to provide additional context. Supports markdown formatting. |
| `required` | boolean | If `true`, the field must have a value before the form can be submitted. Default: `false` |
| `placeholder` | string | Placeholder text shown in the input when empty. Only supported by certain field types. |
| `default_value` | mixed | Default value to use when the field is not provided. Type must match the field type. |
| `grid_size` | integer | Width of the field in a 12-column grid system. Valid values: 1-12. Default: 4 |
| `grid_size_before` | integer | Number of empty grid columns to insert before this field. Valid values: 1-12. |
| `grid_size_after` | integer | Number of empty grid columns to insert after this field. Valid values: 1-12. |
| `advanced` | boolean | If `true`, the field is grouped under "Advanced Settings" and hidden by default. Default: `false` |
| `enum` | boolean | If `true`, the field becomes a dropdown/select with predefined options defined in `enum_options`. Default: `false` |
| `enum_multiple` | boolean | If `true`, allows multiple selections from `enum_options`. Default: `false` |
| `enum_options` | array | Array of option objects for enum fields. See [Enum Options](#enum-options). |
| `min` | number | Minimum value for numeric and date/datetime fields. |
| `max` | number | Maximum value for numeric and date/datetime fields. |
| `max_length` | integer | Maximum character length for text, color, link, and secret fields. |
| `depends_on` | array | Array of dependency conditions that control field visibility. See [Conditional Display](#conditional-display). |

### Advanced Properties

These properties are reserved for advanced use cases by Lucit administrators and are not covered in this specification:

- `component`
- `component_options`
- `policies`

## Field Types

### Numeric Types

#### `int` - Integer

Stores whole numbers.

**Supported Properties:**
- `placeholder` ✓
- `default_value` ✓
- `enum` ✓
- `min` / `max` ✓

**Server Validation:** `integer`

**Example:**
```json
{
  "key": "additional_forecast_days",
  "name": "Additional forecast days",
  "type": "int",
  "required": true,
  "enum": true,
  "enum_options": [
    {"label": "Today", "value": 0},
    {"label": "Today & Tomorrow", "value": 1},
    {"label": "Today plus 2 days", "value": 2}
  ],
  "default_value": 0
}
```

#### `float` - Floating Point Number

Stores decimal numbers.

**Supported Properties:**
- `placeholder` ✓
- `default_value` ✓
- `enum` ✓
- `min` / `max` ✓

**Server Validation:** `numeric`

**Example:**
```json
{
  "key": "temperature_threshold",
  "name": "Temperature Threshold",
  "type": "float",
  "min": -40.0,
  "max": 120.0,
  "default_value": 32.0
}
```

### Text Types

#### `text` - Text String

Stores text input. Can be single-line or multi-line.

**Supported Properties:**
- `placeholder` ✓
- `default_value` ✓
- `enum` ✓
- `max_length` ✓
- `multiline` ✓ (additional property)

**Server Validation:** `string`

**Example:**
```json
{
  "key": "google_sheet_url",
  "name": "Google Sheet URL",
  "type": "text",
  "required": true,
  "placeholder": "https://docs.google.com/spreadsheets/...",
  "description": "Copy and paste the URL to your Google spreadsheet"
}
```

#### `color` - Color Picker

Stores color values (typically hex format).

**Supported Properties:**
- `placeholder` ✓
- `default_value` ✓
- `enum` ✓
- `max_length` ✓

**Server Validation:** `string`

**Example:**
```json
{
  "key": "brand_color",
  "name": "Brand Color",
  "type": "color",
  "default_value": "#FF0000"
}
```

#### `link` - URL

Stores and validates URL strings.

**Supported Properties:**
- `placeholder` ✓
- `default_value` ✓
- `max_length` ✓

**Server Validation:** `url`

**Example:**
```json
{
  "key": "webhook_url",
  "name": "Webhook URL",
  "type": "link",
  "required": true,
  "placeholder": "https://example.com/webhook"
}
```

#### `secret` - Encrypted Secret

Stores sensitive data with automatic encryption. Values are automatically encrypted when entered by users and are scoped to the application.

**Supported Properties:**
- `max_length` ✓
- `allow_show_secret` ✓ (enables show/hide toggle for users with permission)

**Server Validation:** `string`

**Security Note:** Secret fields are automatically encrypted using Lucit's scoped encryption system. Encrypted values are stored in a secure format and can only be decrypted within the proper application scope.

**Example:**
```json
{
  "key": "api_key",
  "name": "API Key",
  "type": "secret",
  "required": true,
  "allow_show_secret": true,
  "description": "Your API key for authentication"
}
```

### Boolean Type

#### `bool` - Boolean

Stores true/false values.

**Supported Properties:**
- `default_value` ✓

**Server Validation:** `boolean`

**Example:**
```json
{
  "key": "enable_caching",
  "name": "Enable Caching",
  "type": "bool",
  "default_value": true
}
```

### Structured Data Types

#### `json` - JSON Object

Stores JSON-formatted data.

**Supported Properties:**
- `default_value` ✓
- `enum` ✓

**Server Validation:** `json`

**Note:** This field type is under active development. Validation and encoding behavior may be refined in future updates.

**Example:**
```json
{
  "key": "custom_config",
  "name": "Custom Configuration",
  "type": "json",
  "default_value": {}
}
```

#### `array` - Array

Stores array data. Often used with `enum_multiple` for multi-select dropdowns.

**Supported Properties:**
- `default_value` ✓
- `enum` ✓

**Server Validation:** `array`

**Note:** This field type is under active development. Validation and encoding behavior may be refined in future updates.

**Example:**
```json
{
  "key": "selected_features",
  "name": "Features",
  "type": "array",
  "enum": true,
  "enum_multiple": true,
  "enum_options": [
    {"label": "Feature A", "value": "feature_a"},
    {"label": "Feature B", "value": "feature_b"}
  ]
}
```

### Date & Time Types

#### `date` - Date

Stores date values.

**Supported Properties:**
- `placeholder` ✓
- `default_value` ✓
- `min` / `max` ✓

**Server Validation:** `date`

**Note:** This field type is under active development. The internal storage format (string, timestamp, etc.) is being finalized.

#### `datetime` - Date and Time

Stores date and time values.

**Supported Properties:**
- `placeholder` ✓
- `default_value` ✓
- `min` / `max` ✓

**Server Validation:** `date`

**Note:** This field type is under active development. The internal storage format (string, timestamp, etc.) is being finalized.

### Display-Only Types

These field types are used for organizing and structuring forms. They do not store data and do not require a `name` property.

#### `header` - Section Header

Displays a header with optional description text.

**Example:**
```json
{
  "key": "header1",
  "name": "Weather Data from Weather API",
  "type": "header",
  "grid_size": 12,
  "description": "For each screen attached to your campaign, add weather data elements to your creatives in the template designer."
}
```

#### `divider` - Visual Separator

Displays a horizontal line to separate form sections.

**Example:**
```json
{
  "key": "divider1",
  "type": "divider",
  "grid_size": 12
}
```

#### `space` - Empty Space

Adds empty vertical space for layout purposes.

**Example:**
```json
{
  "key": "space1",
  "type": "space",
  "grid_size": 12
}
```

## Enum Options

When `enum` is set to `true`, the field displays as a dropdown/select with predefined options. Each option is an object with the following properties:

| Property | Type | Required | Description |
|----------|------|----------|-------------|
| `value` | mixed | Yes | The actual value stored when this option is selected. Type should match the field type. |
| `label` | string | Yes | Display text shown to the user for this option. |
| `description` | string | No | Optional help text for this specific option. |

**Example:**
```json
{
  "key": "routing_preference",
  "name": "Routing Preference",
  "type": "text",
  "enum": true,
  "required": true,
  "enum_options": [
    {
      "label": "Traffic Un-Aware",
      "value": "TRAFFIC_UNAWARE",
      "description": "Calculate routes without considering current traffic"
    },
    {
      "label": "Traffic Aware",
      "value": "TRAFFIC_AWARE",
      "description": "Use current traffic conditions for calculations"
    }
  ],
  "default_value": "TRAFFIC_UNAWARE"
}
```

### Multi-Select Enums

Set `enum_multiple` to `true` to allow selecting multiple options. 

If the field is some scalar (`string`,`int`,`float`) then a csv string will be stored as the result.  If the field is `json` or `array` then we store an object with the results containing an array of the choices


**Example:**
```json
{
  "key": "selected_days",
  "name": "Operating Days",
  "type": "array",
  "enum": true,
  "enum_multiple": true,
  "enum_options": [
    {"label": "Monday", "value": "mon"},
    {"label": "Tuesday", "value": "tue"},
    {"label": "Wednesday", "value": "wed"}
  ],
  "default_value": []
}
```

## Conditional Display

The `depends_on` property allows you to show or hide fields based on the values of other fields. This creates dynamic forms that adapt to user input.

### Structure

`depends_on` is an array of dependency objects. Each dependency has three required properties:

| Property | Type | Description |
|----------|------|-------------|
| `key` | string | The key of the field to check. Must reference another field in the same form. |
| `operator` | string | The comparison operator. See [Available Operators](#available-operators). |
| `value` | mixed | The value to compare against. |

### Evaluation Logic

- **Multiple dependencies are evaluated with AND logic** - ALL conditions must be met for the field to display
- The field is **hidden** (not just disabled) when dependencies are not met
- A field **cannot depend on itself**
- Dependencies can reference any other field in the form, but nested/circular dependencies should be avoided

### Example

```json
{
  "key": "departure_time",
  "name": "Use specific departure time",
  "type": "text",
  "depends_on": [
    {
      "key": "routing_preference",
      "operator": "equals",
      "value": "TRAFFIC_AWARE"
    }
  ],
  "description": "Enter time in HH:MM TZ format (e.g. 3:00pm EST)"
}
```

In this example, the `departure_time` field only appears when `routing_preference` is set to `"TRAFFIC_AWARE"`.

### Available Operators

| Operator | Description |
|----------|-------------|
| `equals` | Value exactly matches |
| `does_not_equals` | Value does not match |
| `is_one_of` | Value is in an array of options |
| `not_any_of` | Value is not in an array of options |
| `>` | Greater than (numeric) |
| `<` | Less than (numeric) |
| `>=` | Greater than or equal to (numeric) |
| `<=` | Less than or equal to (numeric) |
| `is_between` | Value is between two numbers |
| `like` | String contains substring |
| `like_one_of` | String contains one of multiple substrings |
| `like_all_of` | String contains all of multiple substrings |
| `not_like` | String does not contain substring |
| `not_like_any_of` | String does not contain any of multiple substrings |
| `is_true` | Boolean value is true |
| `is_false` | Boolean value is false |
| `is_empty` | Value is empty/null |
| `is_not_empty` | Value has content |
| `matches_regexp` | Value matches regular expression |
| `does_not_match_regexp` | Value does not match regular expression |

### Multiple Dependencies Example

```json
{
  "key": "advanced_options",
  "name": "Advanced Options",
  "type": "text",
  "depends_on": [
    {
      "key": "enable_advanced",
      "operator": "is_true",
      "value": true
    },
    {
      "key": "user_level",
      "operator": "is_one_of",
      "value": ["admin", "power_user"]
    }
  ]
}
```

This field only displays when BOTH conditions are met: `enable_advanced` is `true` AND `user_level` is either "admin" or "power_user".

## Grid Layout System

Lucit Form Fields use a 12-column grid system for responsive layouts, similar to Bootstrap or Material-UI.

### Grid Size

Use `grid_size` to control field width:

```json
{
  "key": "full_width_field",
  "name": "Full Width",
  "type": "text",
  "grid_size": 12  // Takes full width
}
```

```json
{
  "key": "half_width_field",
  "name": "Half Width",
  "type": "text",
  "grid_size": 6  // Takes half width
}
```

```json
{
  "key": "third_width_field",
  "name": "Third Width",
  "type": "text",
  "grid_size": 4  // Takes one-third width (default)
}
```

### Grid Spacing

Use `grid_size_before` and `grid_size_after` to add empty columns around a field:

```json
{
  "key": "centered_field",
  "name": "Centered Field",
  "type": "text",
  "grid_size": 6,
  "grid_size_before": 3,  // 3 empty columns before
  "grid_size_after": 3    // 3 empty columns after
}
```

**Note:** Ensure that `grid_size_before + grid_size + grid_size_after ≤ 12` to avoid layout issues.

## Advanced Fields

Fields marked with `advanced: true` are grouped separately and hidden behind an "Advanced Settings" toggle button. This helps keep forms simple for most users while providing power users access to additional options.

### Requirements for Advanced Fields

If a field is both `advanced` and `required`, it **must** have a `default_value`:

```json
{
  "key": "advanced_setting",
  "name": "Advanced Setting",
  "type": "int",
  "advanced": true,
  "required": true,
  "default_value": 5  // Required because field is both advanced and required
}
```

This ensures that users who never open the advanced settings section still have valid form data.

### Example

```json
{
  "key": "h3_clustering_resolution",
  "name": "Distance From Screen Resolution",
  "type": "int",
  "advanced": true,
  "enum": true,
  "enum_options": [
    {"label": "0.5 Miles", "value": 8},
    {"label": "1.5 Miles", "value": 7},
    {"label": "4 Miles", "value": 6}
  ],
  "default_value": 5,
  "description": "Useful for adjusting performance for 100's or 1000's of screens"
}
```

## Field Macros

Field properties support dynamic value substitution using macros. Macros allow you to reference values from other fields in the form.

### Syntax

Use the format `{Fields.key}` where `key` is the field key you want to reference:

```json
{
  "key": "confirmation_message",
  "name": "Confirmation",
  "type": "header",
  "description": "You selected {Fields.routing_preference} for {Fields.google_sheet_url}"
}
```

If the field is `json` or `array` type, the syntax is `{Fields.key.some_key.some_other_key}` where `some_key.some_other_key` (as an example) are provided by the field.   

### Supported Properties

Macros can be used in any string property, including:

- `name`
- `description`
- `placeholder`
- Any string value within `enum_options`
- Any string value in nested objects/arrays

### Example

```json
[
  {
    "key": "user_name",
    "name": "Your Name",
    "type": "text",
    "required": true
  },
  {
    "key": "greeting",
    "name": "Welcome!",
    "type": "header",
    "description": "Hello, {Fields.user_name}! Please complete the settings below."
  }
]
```

## Validation Rules

Each field type has associated Server validation rules that are automatically applied:

| Field Type | Validation Rule | Additional Validations |
|------------|----------------|----------------------|
| `int` | `integer` | `min`, `max` (if specified) |
| `float` | `numeric` | `min`, `max` (if specified) |
| `text` | `string` | `max_length` (if specified) |
| `color` | `string` | `max_length` (if specified) |
| `secret` | `string` | `max_length` (if specified) |
| `link` | `url` | `max_length` (if specified) |
| `bool` | `boolean` | - |
| `json` | `json` | - |
| `array` | `array` | - |
| `date` | `date` | `min`, `max` (if specified) |
| `datetime` | `date` | `min`, `max` (if specified) |

### Required Fields

When `required: true` is set, the field must have a value before form submission. Required validation is applied in addition to type-specific validation.

### Enum Validation

When `enum: true` is set, the submitted value must match one of the `value` properties in the `enum_options` array.

## Complete Examples

### Weather Data Source Configuration

```json
{
  "fields": [
    {
      "key": "header1",
      "name": "Weather Data from Weather API",
      "type": "header",
      "grid_size": 12,
      "description": "For each screen attached to your campaign, add weather data elements to your creatives in the template designer."
    },
    {
      "key": "additional_forecast_days",
      "enum": true,
      "name": "Additional forecast days",
      "type": "int",
      "required": true,
      "grid_size": 6,
      "description": "How many additional days of forecast data to fetch",
      "enum_options": [
        {"label": "Today", "value": 0},
        {"label": "Today & Tomorrow", "value": 1},
        {"label": "Today plus 2 days", "value": 2},
        {"label": "+3 Days", "value": 3}
      ],
      "default_value": 0
    },
    {
      "key": "limit_weather_keys",
      "name": "Limit weather data to fields",
      "type": "text",
      "advanced": true,
      "required": false,
      "grid_size": 12,
      "description": "A comma separated list of fields to limit the weather data to. Useful for campaigns with 100 or more screens",
      "default_value": ""
    },
    {
      "key": "h3_clustering_resolution",
      "enum": true,
      "name": "Distance From Screen Resolution",
      "type": "int",
      "advanced": true,
      "required": false,
      "grid_size": 6,
      "description": "The radius around a screen to pull a single weather data point for. Useful for adjusting performance for 100's or 1000's of screens",
      "enum_options": [
        {"label": "0.5 Miles", "value": 8},
        {"label": "1.5 Miles", "value": 7},
        {"label": "4 Miles", "value": 6},
        {"label": "10 Miles", "value": 5}
      ],
      "default_value": 5
    }
  ]
}
```

### Drive Time Calculation with Conditional Fields

```json
{
  "fields": [
    {
      "key": "header1",
      "name": "Drive Time to Locations from a Google Spreadsheet",
      "type": "header",
      "grid_size": 12,
      "description": "Make a copy of this [example spreadsheet](https://docs.google.com/spreadsheets/d/1ddZYo3u7VpSl8LT2znYxBmzpPu5fo4p-GRukfGrxojI/) that you can use as a template to get started"
    },
    {
      "key": "google_sheet_url",
      "name": "Google Sheet URL",
      "type": "text",
      "required": true,
      "grid_size": 12,
      "description": "Copy and paste the URL to your Google spreadsheet",
      "placeholder": "https://docs.google.com/spreadsheets/..."
    },
    {
      "key": "googleRoutesApiKey",
      "name": "Google Routes API Key",
      "type": "secret",
      "required": true,
      "grid_size": 12,
      "description": "To use this app, you must register an API key in Google Cloud that has permissions to access the Google Routes API"
    },
    {
      "key": "routingPreference",
      "enum": true,
      "name": "Routing Preference",
      "type": "text",
      "required": true,
      "enum_options": [
        {"label": "Traffic Un-Aware", "value": "TRAFFIC_UNAWARE"},
        {"label": "Traffic Aware (Current Traffic Conditions)", "value": "TRAFFIC_AWARE"}
      ],
      "default_value": "TRAFFIC_UNAWARE"
    },
    {
      "key": "departureTime",
      "name": "Use specific departure time",
      "type": "text",
      "advanced": true,
      "required": false,
      "grid_size": 12,
      "depends_on": [
        {
          "key": "routingPreference",
          "value": "TRAFFIC_AWARE",
          "operator": "equals"
        }
      ],
      "description": "To only get the travel time for a specific departure time, enter it in here in HH:MM TZ format (e.g. 3:00pm EST)",
      "default_value": ""
    }
  ]
}
```

## Best Practices

### Field Keys

- Use descriptive, snake_case keys: `google_sheet_url`, `api_key`, `enable_caching`
- Keys must be unique within a form
- Keep keys consistent across related forms

### Descriptions

- Write clear, concise descriptions that explain the purpose and format
- Include examples when format matters (e.g., "HH:MM TZ format like 3:00pm EST")
- Use markdown for links and formatting
- Keep descriptions user-friendly, not technical

### Default Values

- Always provide sensible defaults for optional fields
- Required advanced fields MUST have default values
- Match the default value type to the field type

### Grid Layout

- Use `grid_size: 12` for headers, dividers, and full-width fields
- Group related fields with consistent grid sizes
- Use half-width (`grid_size: 6`) for paired fields

### Advanced Fields

- Only mark fields as advanced if they're truly optional or for power users
- Always provide default values for advanced fields
- Group related advanced settings together

### Enum Options

- Order options logically (most common first, or alphabetically)
- Use clear, descriptive labels
- Keep values consistent (e.g., uppercase constants or lowercase strings)
- Add descriptions for options that need clarification

### Conditional Display

- Keep dependency chains shallow (avoid field A depends on B depends on C)
- Ensure dependent fields have appropriate defaults
- Test all combinations of conditional display logic
- Document complex conditional logic in field descriptions

## Troubleshooting

### Field Not Displaying

Check:
1. Field has required `key` and `type`
2. Field type requires `name` (all except header/divider/space)
3. `depends_on` conditions are met
4. Field is not marked as `advanced` (check advanced settings toggle)

### Validation Errors

Check:
1. `default_value` type matches field `type`
2. Field type supports the optional properties you're using
3. `enum_options` are properly formatted with `value` and `label`
4. `grid_size` values are between 1 and 12
5. Required fields have values provided

### Enum Not Working

Check:
1. `enum` is set to `true`
2. `enum_options` array exists and is not empty
3. Each option has both `value` and `label`
4. Field type supports enums (int, float, text, color, json, array)

### Conditional Display Issues

Check:
1. Referenced field key exists in the form
2. Operator is valid (see [Available Operators](#available-operators))
3. Value type matches the referenced field's type
4. Field is not depending on itself
5. Multiple conditions are using AND logic (all must be true)
