# Lucit Render App Guide

This guide describes the technical details of the Lucit Render App, its source code, and the rendering process.

## Overview

The Lucit Render App is the application that takes HTML, CSS, and JavaScript from a Lucit template, and then renders the final template in either the Lucit template designer or our Chromium-based snapshotting service.

## Process Overview

The rendering process works like this:

1. The main Render App default HTML document is loaded
2. The HTML, CSS, and JavaScript are injected into the HTML document
3. The resulting HTML is either:
   * Injected into an iframe in the Template Designer
   * Rendered fully in the Chromium-based Snapshotting Service

## HTML

The Render App HTML is as follows:

**`dt_default.html`**
```html
<!DOCTYPE html>
<html lang="en">
  <head>
    <meta charset="UTF-8" />
    <title>LuCore Template Render App</title>
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <style id="lc_dt_styles">{LC_DT_CSS}</style>
  </head>

  <body>
    <div id="container" class="lc_dt_container">
      {LC_DT_HTML}
    </div>
    <script>{LC_DT_JS}</script>
  </body>
</html>
```

You will note that there are three replacement macros in this HTML used to inject the HTML, CSS, and JavaScript from a template into this document:

- `{LC_DT_HTML}`
- `{LC_DT_CSS}`
- `{LC_DT_JS}`

There is a different process used to assemble the code for each one of these replacements.

## Template Code

Each template has the following code (and/or options) stored within it:

- `html` — The HTML for the template
- `css` — The global CSS
- `css_size_variants` — CSS for each [screen format](../../guides/screen-format-reference.md), keyed by the format id (e.g., tall, vertical, bulletin, etc.)
- `fonts` — A list of fonts used in the template
- `js` — The custom JavaScript functions for a template
- `motion_capable` — Whether or not this template has animations or videos

## HTML Injection

The HTML from the template is taken without modification and injected directly into the `{LC_DT_HTML}` macro location.

## CSS Injection

The CSS is formed by combining multiple CSS sources, in the following order:

- `dt_default.css` — The Render App CSS for all templates
- If `motion_capable` is `true`: `dt_animations.css` — Render App CSS animations for all templates
- `fonts` — For each font in the font array, an `@font-face` declaration is added with the full URL to the font file
- `css` — The template CSS
- `css_size_variants` — For each CSS size variant, a media query section is added based on min-aspect-ratio and max-aspect-ratio for the size format

Once the final CSS string from the above items is combined, it is injected directly into the `{LC_DT_CSS}` macro location.

**`dt_default.css`**

```css
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

html {
    height: 100vh
}

body {
    min-height: 100vh;
    height: 100%;
    margin: 0;
    padding: 0;
    box-sizing: border-box
}

/**Containers**/
.lc_dt_container {
    width: 100vw;
    max-width: 100vw;
    height: 100vh;
    max-height: 100vh;
    overflow: hidden;
    position: relative;
}

/**Animations**/

/** All elements inside the container start paused **/
.lc_dt_container * {
    animation-play-state: paused !important;
}

/** When lc_dt_animation_stat class is added to the container, all elements inside the container start playing **/
.lc_dt_container.lc_dt_animation_start * {
    animation-play-state: running !important;
}


/**Images**/

.lc_dt_image {
    position: absolute;
    background-repeat: no-repeat;
}

.lc_dt_video {
    position: absolute;
}

.lc_dt_image_center_cover {
    background-position: center;
    background-size: cover;
}

.lc_dt_image_center_contain {
    background-size: contain;
    background-repeat: no-repeat;
    background-position: center;
}


.lc_dt_image_full {
    background-size: 100% auto;
    background-repeat: no-repeat;
    background-position: center;
}

.lc_dt_image_stretch {
    background-size: 100% 100%;
    background-repeat: no-repeat;
    background-position: center;
}

.lc_dt_image_full_size_stretch {
    width: 100% !important;
    height: 100% !important;
    top: 0 !important;
    left: 0 !important;
    background-size: 100% 100%;
}


/**Text**/
.lc_dt_text {
    position: absolute;
    overflow: hidden;
    display: flex;
    justify-content: center;
    text-align: center;
    align-content: center;
    flex-direction: column;
    /**Just for the initial sizing if not set so you don't lose it**/
    color: #bbbbbb;
    font-family: arial;
    cursor: default;
}

.lc_format_fit_text {
    /**handled by javascript function**/
    font-size: 8px;
    /**An initial font size so we have a starting point**/
}

.lc_format_price_super_script_us .dollars {
    font-size: 0.5em;
    display: inline;
}

.lc_format_price_super_script_us .cents {
    font-size: 0.5em;
    display: inline;
}


/**Objects**/

.lc_dt_object {
    position: absolute;
}

.lc_dt_object_svg {
    position: absolute;
}

.lc_dt_object_svg svg {
    width: 100%;
    height: 100%;
}
```

**`dt_animations.css`**

The `dt_animations.css` code is only included in the render when `motion_capable` is `true` for a specific template.

This file contains a list of pre-defined named `@keyframes` that can be applied to elements using our animation architecture.

For example :

```css

@keyframes flash {

    from,
    50%,
    to {
        opacity: 1;
    }

    25%,
    75% {
        opacity: 0;
    }
}

...More animations

```


## JavaScript Injection

When injecting JavaScript, we combine the following code in this order:

- `dt_default.js` — The Render App JS for all templates
- `js` from the template — The custom JavaScript from the template

This JS is then wrapped in a `try/catch` block and injected directly into the `{LC_DT_JS}` macro location.

## Rendering Process

When the final document is assembled, the following high-level actions take place (driven by `dt_default.js`):

1. Functions are registered that can be used during render
   * Designer functions
   * Text formatting functions
   * Designer formatting functions
2. If video content exists, video sources are pre-fetched and converted to blobs
3. Placeholders are applied
4. Designer functions are evaluated
5. Max character lengths are enforced
6. Formatting functions are executed
7. Text formatting functions are executed
8. Auto-size text fitting is executed
9. Hidden elements are found and marked
10. Once all images are loaded, a `designer_ready` message is sent to the parent frame
11. If auto play is enabled, animations and videos are immediately started
    - Otherwise, we wait for a `designer_play_content` message from the parent


