# Lucit Template CSS Guide

# Overview

Lucit templates use a combination of pre-existing CSS classes plus template defined CSS that is unique to each template

This CSS is combined into a single CSS file that is injected directly into the `head` of the final HTML document at build time.

# Architecture Goals

The goals of the CSS architecture that has been created for Lucit Templates is as follows:

- Will respond to different screen sizes and aspect ratios
- Allow users to customize each screen format independent from the other
- Allow a simple and understandable format that can be edited with WYSIWYG tools in the Lucit Template Designer


# Global CSS

There are 2 global CSS files.  See [Render App Guide](./lucit-render-app-guide.md) for details.

- `dt_default.css` Applies to all templates and provides basic layout and positioning
- `dt_animations.css` Optional and only added if  `motion_capable` is `true` and contains a list of `keyframes` that can be applied to animations on an element.

# Template CSS

The Template CSS is broken into 2 parts

- `css` the css that applies to all sizes.  These are only used when the template Editor Mode is set to `All Sizes`
- `css_size_variants` specific css for each size format that is applied through the use of media queries.   These size variants are only used when the template editor mode is set to `Each Size Independently`


# Example Template

In this example template, we have 3 sizes (Vertical, Tall and Widescreen) and Editor Mode is set to `All Sizes`

In our code editor, we see the following

**HTML**

```html
<div
  id="rectangle_default_bg"
  title=""
  x-objectcode="rectangle"
  class="lc_ut_designer lc_dt_object lc_dt_default_background"
  x-fieldname="rectangle"
  x-optional-field="false"
  x-element-settings="{}"
  x-element-setting-values="{}"></div>

<div
  id="obj_your_new_text_ra0dsgwj"
  title="Your New Text"
  x-objectcode="your_new_text"
  class="lc_ut_designer lc_dt_element lc_dt_text lc_dt_text_editable lc_format_fit_text"
  style=""
  x-fieldname="your_new_text"
  x-optional-field="false"
  x-element-settings="{}"
  x-element-setting-values="{}">
  Your New Text
</div>

<div
  id="obj_rectangle_9czjtra1"
  title="Rectangle"
  x-objectcode="rectangle"
  class="lc_ut_designer lc_dt_element lc_dt_object"
  style=""
  x-fieldname="rectangle"
  x-optional-field="false"
  x-element-settings="{}"
  x-element-setting-values="{}"></div>

```

**CSS**

```css
#rectangle_default_bg {
  width: 100%;
  height: 100%;
  top: 0%;
  left: 0%;
  background: white;
}

#obj_your_new_text_ra0dsgwj {
  white-space: pre-line;
  overflow: visible;
  width: 61.4486%;
  height: 39.9985%;
  top: 50%;
  left: 19.1589%;
  font-family: "Public Sans Thin";
  color: #4a90e2;
  rotate: none;
  text-shadow: 3px 3px 5px rgba(0, 0, 0, 0.45);
}

#obj_rectangle_9czjtra1 {
  background-color: #333333;
  width: 69.3889%;
  height: 29.2937%;
  top: 9.1423%;
  left: 15.1869%;
  rotate: none;
  background: linear-gradient(
    90deg,
    rgba(213, 11, 11, 1) 0%,
    RGBA(108, 22, 22, 1) 100%
  );
}

```


As you can see, the editor applies css styles to elements based on the `id` of the element.   Every new element that is added to the template, will receive a new CSS style block for its `id`

## Positioning

All positioning is handled with relative units at 4 decimal places (percentages) for `width`, `height`, `top` and `left`

As the user drags / moves an element around on the canvas, or, resizes the element, the editor updates these 4 css styles for the element during this user action

In addition, css styles like background, or text-shadow, etc., are all updated via the available tools on the `Edit` tab of the Template Designer


## Render

When this template is rendered via the Render App, the resulting CSS looks like this

**Editor Mode : All Sizes**

```css
// ...existing code...

/** dt_default.css **/
/** //all of the global dt_default.css **/
/** FontCSS **/
/** If custom fonts were used, they will appear here **/
/** TemplateCSS **/
#rectangle_default_bg {
  width: 100%;
  height: 100%;
  top: 0%;
  left: 0%;
  background: white;
}

#obj_your_new_text_ra0dsgwj {
  white-space: pre-line;
  overflow: visible;
  width: 61.4486%;
  height: 39.9985%;
  top: 50%;
  left: 19.1589%;
  font-family: "Public Sans Thin";
  color: #4a90e2;
  rotate: none;
  text-shadow: 3px 3px 5px rgba(0, 0, 0, 0.45);
}

#obj_rectangle_9czjtra1 {
  background-color: #333333;
  width: 69.3889%;
  height: 29.2937%;
  top: 9.1423%;
  left: 15.1869%;
  rotate: none;
  background: linear-gradient(
    90deg,
    rgba(213, 11, 11, 1) 0%,
    rgba(108, 22, 22, 1) 100%
  );
}

// ...existing code...

```

If the user then switches the Editor Mode of this template to `Each Size Independently` and makes adjustments on each size.

The individual css changes for each size are stored in the `css_size_variants` keyed on the size name id (e.g. `widescreen` or `vertical`) and then in the CSS are wrapped in media queries based on the aspect ratio of the screen format.  See [Screen Format Reference](../../guides/screen-format-reference.md)

When this template is rendered, the resulting CSS looks like this

**Editor Mode : Each Size Independently**

```css
// ...existing code...
/** dt_default.css **/
/** //all of the global dt_default.css **/
/** FontCSS **/
/** If custom fonts were used, they will appear here **/
/** TemplateCSS **/
#rectangle_default_bg {
  width: 100%;
  height: 100%;
  top: 0%;
  left: 0%;
  background: white;
}

#obj_your_new_text_ra0dsgwj {
  white-space: pre-line;
  overflow: visible;
  width: 61.4486%;
  height: 39.9985%;
  top: 50%;
  left: 19.1589%;
  font-family: "Public Sans Thin";
  color: #4a90e2;
  rotate: none;
  text-shadow: 3px 3px 5px rgba(0, 0, 0, 0.45);
}

#obj_rectangle_9czjtra1 {
  background-color: #333333;
  width: 69.3889%;
  height: 29.2937%;
  top: 9.1423%;
  left: 15.1869%;
  rotate: none;
  background: linear-gradient(
    90deg,
    rgba(213, 11, 11, 1) 0%,
    rgba(108, 22, 22, 1) 100%
  );
}

/** SizeVariantsCSS **/
/* BOARD FORMAT MEDIA QUERY : Tall */
@media (min-aspect-ratio: 600/1000) and (max-aspect-ratio: 799/1000) {
  #rectangle_default_bg {
    width: 100%;
    height: 100%;
    top: 0%;
    left: 0%;
    background: white;
  }
  #obj_your_new_text_ra0dsgwj {
    white-space: pre-line;
    overflow: visible;
    width: 61.4486%;
    height: 39.9985%;
    top: 38.3771%;
    left: 15.1869%;
    font-family: "Public Sans Thin";
    color: #4a90e2;
    rotate: none;
    text-shadow: 3px 3px 5px rgba(0, 0, 0, 0.45);
  }
  #obj_rectangle_9czjtra1 {
    background-color: #333333;
    width: 69.3889%;
    height: 29.2937%;
    top: 2.7593%;
    left: 11.2186%;
    rotate: none;
    background: linear-gradient(
      90deg,
      rgba(213, 11, 11, 1) 0%,
      rgba(108, 22, 22, 1) 100%
    );
  }
}
/* END BOARD FORMAT MEDIA QUERY : Tall */

/* BOARD FORMAT MEDIA QUERY : Vertical */
@media (min-aspect-ratio: 400/1000) and (max-aspect-ratio: 599/1000) {
  #rectangle_default_bg {
    width: 100%;
    height: 100%;
    top: 0%;
    left: 0%;
    background: white;
  }
  #obj_your_new_text_ra0dsgwj {
    white-space: pre-line;
    overflow: visible;
    width: 82.8488%;
    height: 69.5581%;
    top: 20.0440%;
    left: 10.2653%;
    font-family: "Public Sans Thin";
    color: #4a90e2;
    rotate: none;
    text-shadow: 3px 3px 5px rgba(0, 0, 0, 0.45);
  }
  #obj_rectangle_9czjtra1 {
    background-color: #333333;
    width: 86.6279%;
    height: 12.2750%;
    top: 3.5751%;
    left: 8.2077%;
    rotate: none;
    background: linear-gradient(
      90deg,
      rgba(213, 11, 11, 1) 0%,
      rgba(108, 22, 22, 1) 100%
    );
  }
}
/* END BOARD FORMAT MEDIA QUERY : Vertical */

/* BOARD FORMAT MEDIA QUERY : Widescreen */
@media (min-aspect-ratio: 1700/1000) and (max-aspect-ratio: 1800/1000) {
  #rectangle_default_bg {
    width: 100%;
    height: 100%;
    top: 0%;
    left: 0%;
    background: white;
  }
  #obj_your_new_text_ra0dsgwj {
    white-space: pre-line;
    overflow: visible;
    width: 61.4464%;
    height: 40.7083%;
    top: 52.3063%;
    left: 19.1761%;
    font-family: "Public Sans Thin";
    color: #4a90e2;
    rotate: none;
    text-shadow: 3px 3px 5px rgba(0, 0, 0, 0.45);
  }
  #obj_rectangle_9czjtra1 {
    background-color: #333333;
    width: 95.005%;
    height: 56.3083%;
    top: 0.9686%;
    left: 2.3992%;
    rotate: none;
    background: linear-gradient(
      90deg,
      rgba(213, 11, 11, 1) 0%,
      rgba(108, 22, 22, 1) 100%
    );
  }
}
/* END BOARD FORMAT MEDIA QUERY : Widescreen */
// ...existing code...

```

## Template Desiger Managed Styles

The list of styles that is currently managed / edited by the template designer can be found in [Template Designer Managed CSS Styles](./template-css-styles.md)


## Customizing the CSS

Although these CSS styles are designed to be edited by the Lucit Template Designer, the application supports any CSS styles that are available to the Chromium Web Browser

Because of this, developers building custom templates with custom functionality can add any CSS styles that they would like to their templates with the following caveats

- If the style does not have a [corresponding tool in the Edit panel](./template-css-styles.md), it will not be user editable
- If the style does not apply directly to the containing `div` element, but is a style applied to some child elements of the `div`, this style will not be user editable.
- If you override the positioning with non relative units, the template will become unstable, and not editable within the template editor.  (Positions MUST be in percentages)


## Animations

When building animations, the recommended way is to add custom keyframes to your css and then apply those in the editor and edit their settings.

The Animations Architecture actually stores animations as an array of JSON objects in the `x-animations` attribute of the element.   This is so that the template render app has full control of when animations start and stop.  Adding animations directly or outside of this architecture will result in editing and rendering problems as animations may begin or end at the wrong times.

Learn more about Animations with the [Lucit Template Animation Guide](./lucit-template-animation-guide.md)

