---
title: Lucit Template Custom Fields
description: How to design, map, and compute custom fields for Lucit post templates, including Text vs Image fields, Field Map and inventory_attributes, Select (enum) fields, built-in functions, and registering custom designer functions.
---

## Overview

Lucit Template Custom Fields let you compute dynamic values (text or image URLs) for use in your post templates, collect user inputs during Post Ad, and persist those values onto items via `inventory_attributes`. This guide is written for template designers and developers, with concise, implementation-focused examples that search engines and LLMs can parse easily.

What you’ll learn:

* Field types: Text vs Image, and immutability rules
* Field Code vs Field Map, and how values write to `item.inventory_attributes`
* Built-in functions: `fnValue`, `fnConcat`, `fnJson`
* Creating Image-returning fields
* Creating Select (enum) fields for Post Ad
* Auto-creating `inventory_attributes` via Field Map
* Custom element settings and `elementSettings`
* Registering custom designer functions with `registerDesignerFunction`
* Testing in the Post Ad flow and common constraints


## Adding a new Custom Field

To add a new custom field to your template

From your template, under Dynamic Data Elements, scroll down to the Custom Fields section, and click the `+` icon next to `Custom Field` to add a new custom field to your canvas.

This will bring up the Custom Field dialog with the following options

* **Custom Field Name** A name for your field
* **Placeholder** The placeholder that appears when no data is set
* **Text / Image** Choose if this field will create a Text Element or an Image Element on your canvas
* **FIELD CODE** Add the field code using an available Designer Function
* **FIELD MAP** Create a custom field form
* **HELP** Reference and list of available macros


## Concepts

### Field Types

* Text field: returns any text value (string, number, formatted text). Use for text elements.  Under the hood, this creates an element with the `lc_dt_text` css class
* Image field: must return a valid URL to an image. Use for Image Elements.  Under the hood, this creates an element with the `lc_dt_image` css class


Note:  See [HTML Guide](./lucit-template-html-guide.md) for more information on Text and Image fields

Important: After a custom field is created as Text or Image, its type cannot be changed. To switch types, create a new field.

### Field Code vs Field Map

* Field Code: a small program that produces the value for the field, using built-in or custom functions (e.g., `fnValue`, `fnConcat`).
* Field Map: a JSON array declaring fields that should be mapped to `item.inventory_attributes`. When the user posts an ad, the user is presented with fields requesting the values and the mapped field values are stored on the item in the `inventory_attributes` array.

Contract:

* Field Map entry keys: `key` (string), `name` (string), `type` (string). Optional: `enum` (boolean), `enum_options` (array), etc.
* Values write to `item.inventory_attributes[<key>]` during the post process.

Example Field Map (text attribute):

```json
[
	{
		"key": "your_field_key", 
		"type": "text", 
		"name": "Your Field Name"
	}
]
```

Valid attributes and values for a field can be found in the [Lucit Form Fields Specification](../fields/lucit-form-fields-specification.md)

Referencing the mapped value in Field Code:

```javascript
fnValue('{{item.inventory_attributes.your_field_key}}')
```

## Built-in Functions

These functions are available in the Field Code editor:

* `fnValue(input)` :
	* Inputs: a string or macro expression. Typical usage: `fnValue('{{item.inventory_attributes.key}}')`.
	* Output: returns the value resolved from the input; often a simple passthrough of a macro-resolved value.
	* Errors: if the macro is missing, returns empty string;

* `fnConcat(params)`:
	* Inputs: array of parts (strings/macros). Example: `fnConcat(['https://img/', '{{item.unique_id}}', '.png'])`.
	* Output: concatenated string.
	* Errors: non-string parts are coerced to strings.

* `fnJson(obj)`:
	* Inputs: a JSON object or value.
	* Output: stringified JSON.
	* Use when you need to emit structured content into a text field, or serialize data to store in `inventory_attributes`.

Parameter conventions:

* `fnValue` accepts a single value.
* `fnConcat` accepts a single array parameter of parts.
* `fnJson` accepts a single object/value.

Note: You can add new functions via `registerDesignerFunction` (see below).

## Creating an Image-returning Custom Field

Steps:

1) In Designer, go to Dynamic Data Elements → Custom Fields, click + to add a field.
2) Set Type to Image.
3) In the FIELD CODE tab, return a URL string to an image.
4) Place the field on the canvas as needed.

Example (weather icon by condition code):
fnConcat(['https://www.yourplace.com/icons/weather/', '{{digital_board.store.Weather1_current_condition_code}}', '.png'])

This field will then become an Image Element on the canvas and the url will be the value of the FIELD CODE function.

Constraint: Image fields must resolve to a valid image URL; if the URL is invalid or empty, the image may fail to render.

## Creating a Select (enum) Field for Post Ad

You can show a dropdown to the user during Post Ad by using a mapped field with `enum` and `enum_options`.

Steps:

1. Add a new custom field; choose a key (e.g., `my_cool_dropdown`).
2. FIELD CODE:
	 fnValue('{{item.inventory_attributes.my_cool_dropdown}}')
3. FIELD MAP (array with one object):
[
	{
		"key": "my_cool_dropdown",
		"enum": true,
		"name": "This is very cool",
		"type": "text",
		"enum_options": [
			{"label": "This is the first option", "value": "First Option", "description": "Choose this option, for regular stuff", "icon": "https://lucore-bucket-images-prod2.s3.us-east-2.amazonaws.com/2276/img_667c5296ee727_c40c97b51909218e7556.png"},
			{"label": "This is the second option", "value": "Second Option", "description": "Choose this option for much better stuff", "icon": "https://lucore-bucket-images-prod2.s3.us-east-2.amazonaws.com/5/img_633efc8887451_e9defe9717c573356c1a.jpg"}
		]
	}
]
4. Save. The field appears on the canvas and in Post Ad as a dropdown.

Validation rules:

* `enum_options[].label` and `enum_options[].value` are required.
* `enum_options[].description` and `enum_options[].icon` are optional.
* No size limits or URL constraints beyond standard URL validity.

## Auto-creating inventory_attributes

Custom fields can create new `item.inventory_attributes` on the fly in order to provide an unlimited number of text fields during the post ad process (beyond just `item.title` and `item.description`)

Rule: The Field Map `key` must exactly match the attribute name used in Field Code.

Example (notes attribute):

FIELD CODE:
fnValue('{{item.inventory_attributes.notes}}')

FIELD MAP:
[
	{"key": "notes", "name": "Notes", "type": "text"}
]

During Post Ad, the system will request the value and store it at `item.inventory_attributes.notes`.

## Registering Custom Designer Functions

You can extend the Code Editor with your own functions. Use `registerDesignerFunction(name, fn)` to add callable helpers for Field Code and designer formatting.

Contract:

* Inputs: `name` (string), `fn` (function)
* Function Name Rule: The function MUST begin with the 2 characters `fn`  For example `fnMyFunction` is valid, `myFunction` is not.
* IF the function requires multiple arguments, those MUST be passed as an array.   In other words, these functions can only accept one value.  So, if your function requires multiple arguments (for instance, see `fnConcat` then, these must be passed as an array)
* Behavior: Registers `fn` under `name` so it can be called in Field Code. Custom functions MUST accept a single array parameter (e.g., `params`).
* Constraints: Prefer pure, deterministic functions; avoid side effects; handle null/undefined inputs defensively.

Example: sanitize and clamp text length

```javascript
registerDesignerFunction('fnClampText', (params) => {

    const [str,maxLen] = params;

	try {
		const s = (str ?? '').toString();
		const n = Number(maxLen) || 0;
		if (n <= 0) return '';
		return s.length <= n ? s : s.substring(0, n);
	} catch (e) {
		return '';
	}
});
```

Usage in Field Code:

```javascript
fnClampText(['{{item.inventory_attributes.headline}}', 40])
```

Example: safe numeric division with default

```javascript
registerDesignerFunction('fnSafeDivide', (params) => {
    const [a, b, defaultValue] = params
	const x = Number(a);
	const y = Number(b);
	if (!isFinite(x) || !isFinite(y) || y === 0) return defaultValue;
	return x / y;
});
```

Usage in Field Code:

```javascript
fnSafeDivide(['{{item.inventory_attributes.budget}}', '{{item.inventory_attributes.days}}', 0])
```


Guidelines:

* Naming: prefix custom functions with `fn`.
* Inputs: treat macros as strings and coerce types defensively.
* Errors: catch exceptions and return safe defaults.
* Purity: avoid network calls or DOM manipulation; compute-only.

## HTML Examples

The following are examples of the HTML that is created when adding custom field elements to your canvas

**Example using `fnConcat`**

```html

<div
  id="data_source_text_cmcjynmi"
  title=""
  x-objectcode="custom_field_sqizp2qr"
  class="lc_ut_designer lc_dt_data lc_dt_text lc_format_fit_text lc_dt_data_function"
  x-placeholder="2010 Ford Focus"
  x-optional-field="false"
  x-fieldname="Make, Model Year"
  data-value="{fnConcat([
'{{item.year}}',
' ',
'{{item.make}}',
' ',
'{{item.model}}',
]);}"
  x-element-settings="{}"
  x-element-setting-values="{}">
  {fnConcat([ '{{item.year}}', ' ', '{{item.make}}', ' ', '{{item.model}}', ]);}
</div>

```

**Example with custom function `fnClampText`**

```html

<div
  id="data_source_text_fvo14w4k"
  title=""
  x-objectcode="custom_field_so0jvgoe"
  class="lc_ut_designer lc_dt_data lc_dt_text lc_format_fit_text lc_dt_data_function"
  x-placeholder=""
  x-optional-field="false"
  x-fieldname="Clamp to 10 Chars"
  data-value="{fnClampText([
'{{item.title}}',
10
]);}"
  x-element-settings="{}"
  x-element-setting-values="{}">
  {fnClampText([ '{{item.title}}', 10 ]);}
</div>
```


## Testing in Post Ad and Canvas

Quick checks:

* Canvas: place and align your field; ensure the rendered value looks correct.
* Post Ad: open the Post Ad dialog; confirm field presence and input behavior (especially for `enum`).
* Persisted data: verify `item.inventory_attributes` contains the expected keys/values post-submission.
* Image fields: confirm URLs resolve and images load.

## Edge Cases and Tips

* Field Type is immutable. Choose Text vs Image correctly before saving.
* Macro resolution: if a macro key is missing, functions should degrade gracefully (empty string or fallback).
* Image URL validity: return a complete, accessible URL.
* Enum defaults: consider setting a sensible default in your workflow if the user doesn’t select a value.
* JSON storage: when storing structured data, prefer `fnJson({...})` to keep `inventory_attributes` values consistent.

## Summary

Custom Fields power dynamic content in Lucit templates. Use Text fields for text/number outputs, Image fields for image URLs. Map fields with Field Map to write to `item.inventory_attributes`, and enrich user interactions with Select (enum) dropdowns. Extend capabilities by registering your own functions via `registerDesignerFunction`, keeping them pure, defensive, and easy to reuse.

## Related Documents

* **Form Fields** [Lucit Form Fields Specification](../fields/lucit-form-fields-specification.md)
* **Macro Reference** [Lucit Macro Reference](../macros/field-map-macro-reference.md)

