# Lucit Text Formatting Functions Guide

## Overview

Text formatting functions in Lucit let you add custom formatting to text elements using JavaScript functions that are created specifically to format text.

When you add a Text element to your canvas, on the Edit tab, in the Effects & Others section, you will see a `Format` Drop Down that contains a list of ALL available text formatting functions.

By default, Lucit (Render App) comes with a number of [Pre-Built Text Formatting Functions](./lucit-text-formatting-function-reference.md) that are available to all templates for formatting prices, dates, numbers, etc. Use that reference to discover existing formats before creating a custom one.

## How Formatting Functions Work

Formatting functions are applied to elements based on CSS classes. Built-in formatting functions are named exactly the same as their matching CSS class, and they all start with the prefix `lc_format_`.

For instance, the function `lc_format_price_us` is defined in Render App and formats a number like `$1234.56`. This function is applied to any text element that has the `lc_format_price_us` class.

**Example Text Element with `lc_format_price_us`**

Note how this element has the class `lc_format_price_us`.

```html
<div
  id="data_source_text_4hwt8gkb"
  title=""
  x-objectcode="account_store_CoinPaprika1_bitcoin_price"
  class="lc_ut_designer lc_dt_data lc_dt_text lc_format_fit_text lc_format_price_us"
  x-placeholder="87302.192738393"
  x-optional-field="false"
  x-fieldname="Bitcoin Price"
  data-value="{account.store.CoinPaprika1_bitcoin_price}"
  x-group-parent-id="LCUID-LF-8b9ef63b-41db-4811-b747-56938963c691"
  x-element-settings="{}"
  x-element-setting-values="{}">
  {account.store.CoinPaprika1_bitcoin_price}
</div>
```

When a template renders, each formatting function looks for any elements that have the associated class, and then the formatting function executes.

Text formatting functions can operate directly on the HTML DOM element and are also provided the data value of the element.

## Creating Custom Text Formatting Functions

To create custom text formatting functions, you add them in the JS (JavaScript) tab in the Code Editor at `Templates → {Template} → Canvas → Actions Bar → <>`.

Code in the JS tab runs when the template is loaded in the Render App and is scoped to that template.

To register a custom text formatting function, use `registerTextFormattingFunction` with the following three arguments:

- `name`: a name for the function. Alphabetic characters only; no spaces, special characters, or numbers. The `name` is used directly in the generated CSS class, so it must be a valid CSS class fragment.
- `friendlyName`: a label that appears in the Format dropdown in the Edit panel. Can contain spaces and special characters.
- `fn`: the function that will be executed. This function accepts two arguments:
  - `el`: the DOM element.
  - `dataValue`: the value of the `data-value` attribute after macro replacements (or the `x-placeholder` value when macros have not been replaced). For example, if `data-value` is `{item.price}`, then `dataValue` will be the resolved numeric price.

For instance, this function will format a price in Euros:

```javascript
registerTextFormattingFunction(
  "formatPriceEuro",
  "€12,345",
  (el, dataValue) => {
    const numeric = Number(dataValue);
    if (!isFinite(numeric)) {
      el.innerHTML = "";
      return;
    }

    el.innerHTML = "€" + numeric.toLocaleString();
  }
);
```

When the template loads, this will register as an available formatting function in the Format dropdown under the heading `User Formats`.

When you apply this function to a text element on the canvas, it adds a special class `lc_format_tff_formatPriceEuro`.

All custom text formats are prefixed with `lc_format_tff_` followed by the function name (for example, `formatPriceEuro`).

In other words, a text formatting function with the name `formatPriceEuro` will use a class called `lc_format_tff_formatPriceEuro`, and any element with that class will be processed by this function.

**Modify Styles**
Text Formatting functions are very powerful since they can modify the element directly.  Here is an example foramtting function that will make the price field  Bold and Red IF the price is less thatn 10,000

It will also format the value as a price

```javascript
registerTextFormattingFunction(
  "priceBoldRedIfUnder10k",
  "Bold Red if < $10,000",
  (el, dataValue) => {
    const numeric = Number(dataValue);

    // If not a valid number, do nothing
    if (!isFinite(numeric)) {
      return;
    }

    // Format the inner HTML as a price
    const formattedPrice = numeric.toLocaleString(undefined, {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2
    });
    el.innerHTML = "$" + formattedPrice;

    // Apply conditional styling based on the price value
    if (numeric < 10000) {
      el.style.color = "red";
      el.style.fontWeight = "bold";
    } else {
      // Optional: reset styles for values >= 10000
      el.style.color = "";
      el.style.fontWeight = "";
    }
  }
);
```




## HTML Examples

Here are some example HTML elements with text formatting functions applied to them.

**Price, Short US**

```html
<div
  id="data_source_text_4hwt8gkb"
  title=""
  x-objectcode="account_store_CoinPaprika1_bitcoin_price"
  class="lc_ut_designer lc_dt_data lc_dt_text lc_format_fit_text lc_format_price_short_us"
  x-placeholder="87302.192738393"
  x-optional-field="false"
  x-fieldname="Bitcoin Price"
  data-value="{account.store.CoinPaprika1_bitcoin_price}"
  x-group-parent-id="LCUID-LF-8b9ef63b-41db-4811-b747-56938963c691"
  x-element-settings="{}"
  x-element-setting-values="{}">
  {account.store.CoinPaprika1_bitcoin_price}
</div>
```

**Custom Euro Text Formatting Function**

```html
<div
  id="data_source_text_4hwt8gkb"
  title=""
  x-objectcode="account_store_CoinPaprika1_bitcoin_price"
  class="lc_ut_designer lc_dt_data lc_dt_text lc_format_fit_text lc_format_tff_formatPriceEuro"
  x-placeholder="87302.192738393"
  x-optional-field="false"
  x-fieldname="Bitcoin Price"
  data-value="{account.store.CoinPaprika1_bitcoin_price}"
  x-group-parent-id="LCUID-LF-8b9ef63b-41db-4811-b747-56938963c691"
  x-element-settings="{}"
  x-element-setting-values="{}">
  {account.store.CoinPaprika1_bitcoin_price}
</div>
```



